package itineraryfinder;

import com.byteowls.jopencage.JOpenCageGeocoder;
import com.byteowls.jopencage.model.JOpenCageForwardRequest;
import com.byteowls.jopencage.model.JOpenCageResponse;
import com.byteowls.jopencage.model.JOpenCageResult;
import com.jfoenix.controls.JFXButton;
import com.jfoenix.controls.JFXDialog;
import com.jfoenix.controls.JFXDialogLayout;
import com.jfoenix.controls.JFXListView;
import com.jfoenix.controls.JFXTextField;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.ResourceBundle;
import javafx.beans.binding.Bindings;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.fxml.Initializable;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.ListCell;
import javafx.scene.control.MenuItem;
import javafx.scene.layout.StackPane;
import javafx.scene.text.Text;
import java.util.LinkedHashMap;
import javafx.application.HostServices;
import javafx.application.Platform;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.concurrent.Task;
import javafx.scene.control.Hyperlink;
import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.GridPane;
import org.controlsfx.dialog.ProgressDialog;
import salesmansolver.*;

/**
 *
 * @author Andrea Barbagallo
 */
public class ItineraryFinderController implements Initializable {
    
    private final JOpenCageGeocoder geocoder = new JOpenCageGeocoder("81967e28ae5a4d47858df3a1eb737805");
    public HostServices hostServices;
    String startingCity;
    
    @FXML
    private StackPane stackPane;
    
    @FXML
    JFXTextField searchTextField;
    
    @FXML
    JFXTextField iterationsNumberField;
    
    @FXML
    JFXTextField mutationRateField;
    
    @FXML
    JFXButton searchButton;
    
    @FXML
    JFXButton findButton;
    
    @FXML
    JFXButton clearButton;
    
    @FXML
    JFXButton clearCitiesButton;
    
    @FXML
    JFXButton showOnMapButton;
    
    @FXML
    JFXListView placeListView;
    
    private ObservableList<String> observablePlaceList;
    private final ArrayList<Place> placeList = new ArrayList<>();
    
    @FXML
    JFXListView suggestionsListView;
    
    private ObservableList<String> observableSuggestionsList;
    private final ArrayList<Place> placeSuggestions = new ArrayList<>();
    
    @FXML
    Text cityNumberText;
    
    @FXML
    Text startingCityText;
    
    @FXML
    Text distanceNumberText;
    
    @FXML
    Text finalDistance;
    
    @FXML
    public void clearCitiesPressed() {
        observablePlaceList.clear();
        placeList.clear();
        placeListView.setItems(observablePlaceList);
        cityNumberText.setText("0");
        setStartingCity(-1);
        showOnMapButton.setDisable(true);
    }
    
    @FXML
    public void findButtonPressed() {
        if (observablePlaceList.size() < 3)
            return;
        
        int iterations = 1000;
        
        try {
            iterations = Integer.parseInt(iterationsNumberField.getText());
        } catch (NumberFormatException e) {
            System.out.println("No valid iterations number, using 1000");
        }
        
        double mutation_rate = Double.min(1/observablePlaceList.size(), 0.01d);
        
        try {
            mutation_rate = Double.parseDouble(mutationRateField.getText());
            mutation_rate /= 100;
            System.out.println("Mutation rate: " + mutation_rate);
        } catch (NumberFormatException e) {
            System.out.println("Using the default mutation rate");
        }
        
        // In the constructor is passed the maxPopulation
        Population p = new Population(200);
        LinkedHashMap<Integer, Gene> l = new LinkedHashMap<>();
        LinkedHashMap<String, Place> placeByName = new LinkedHashMap<>();
        
        for (int i = 0; i < placeList.size(); i++) {
            l.put(i, new Gene(placeList.get(i).address, placeList.get(i).latLng.lat, placeList.get(i).latLng.lng));
            placeByName.put(placeList.get(i).address, placeList.get(i));
        }
        
        Task task = createTaskItinerary(p, l, iterations, mutation_rate);

        ProgressDialog dialog = new ProgressDialog(task);
        //dialog.setContentText("Searching for cities...");
        dialog.setTitle("Searching...");
        dialog.setHeaderText("Searching for the best itinerary...");

        new Thread(task).start();
        dialog.showAndWait();
        
        // Change order in the GUI
        placeList.clear();
        observablePlaceList.clear();
        
        Chromosome bestPath = p.bestPath();
        finalDistance.setText(new DecimalFormat("#.##").format(bestPath.getRouteDistance()) + " Km");
        ArrayList<Place> tmpPlace = new ArrayList<>();
        ArrayList<String> tmpString = new ArrayList<>();
        int index = -1;
        
        for (int i = 0; i < bestPath.size(); i++) {
            if (startingCity.equals(bestPath.get(i).toString())) {
                index = i;
                break;
            }
            
            tmpPlace.add(placeByName.get(bestPath.get(i).toString()));
            tmpString.add(bestPath.get(i).toString());
        }
        
        if (index >= 0) {
            
            for (int i = index; i < bestPath.size(); i++) {
                placeList.add(placeByName.get(bestPath.get(i).toString()));
                observablePlaceList.add(bestPath.get(i).toString());
            }
            
            placeList.addAll(tmpPlace);
            observablePlaceList.addAll(tmpString);

            placeListView.setItems(observablePlaceList);

            System.out.println("FINAL BEST PATH: " + p.bestPath());
            System.out.println("FINAL WORST PATH: " + p.worstPath());
        }
        else {
            System.err.println("Starting city not setted!");
        }
        
    }
    
    private Task createTaskItinerary(Population p, LinkedHashMap<Integer, Gene> l, int iterations, double mutation_rate) {
        return new Task() {
            @Override
            protected Void call() throws Exception {
                
                p.initialPopulation(l);
                
                updateMessage("Number of generations: " + 0 + "\n"
                                    + "Best path: " + new DecimalFormat("#.##").format(p.bestPath().getRouteDistance()) + " Km"
                                    + ", Current gen: " + new DecimalFormat("#.##").format(p.getCurrentGenerationBestPath()) + "Km\n  ");
                
                Thread.sleep(500);
                
                for (int i = 0; i < iterations; i++) {
                    // Mutation rate usually is 1/chromosomeSize (here 1/20 = 0.05)
                    // The minimum is 0.01
                    updateMessage("Number of generations: " + i + "\n"
                                    + "Best path: " + new DecimalFormat("#.##").format(p.bestPath().getRouteDistance()) + " Km"
                                    + ", Current gen: " + new DecimalFormat("#.##").format(p.getCurrentGenerationBestPath()) + "Km\n  ");
                    
                    updateProgress(i, iterations);
                    
                    // ...(selectionSize, eliteSize, mutationRate);
                    p.nextGeneration(200, 50, mutation_rate);
                }
                
                return null;
            }
        };
    }
    
    @FXML 
    public void clearButtonPressed() {
        clearSuggestions();
    }
    
    @FXML
    public void searchButtonPressed() {
        clearSuggestions();
        setSuggestions(searchTextField.getText());
    }
    
    @FXML
    public void onEnter() {
        clearSuggestions();
        setSuggestions(searchTextField.getText());
    }
    
    @FXML
    public void suggestionClicked(MouseEvent event) {
        
        if(event.getButton().equals(MouseButton.PRIMARY)){
            if(event.getClickCount() == 2){                
                if (suggestionsListView.getSelectionModel().isEmpty())
                    return;
                
                int i = suggestionsListView.getSelectionModel().getSelectedIndex();
                
                if (!observablePlaceList.contains(placeSuggestions.toString())) {
                    observablePlaceList.add(placeSuggestions.get(i).address);
                    placeList.add(placeSuggestions.get(i));
                    
                    if (placeList.size() == 1) {
                        setStartingCity(0);
                    }
                    
                    cityNumberText.setText(String.valueOf(Integer.parseInt(cityNumberText.getText())+1));
                }
                
                placeListView.setItems(observablePlaceList);
                showOnMapButton.setDisable(false);
            }
        }
    }
    
    @Override
    public void initialize(URL url, ResourceBundle rb) {
        iterationsNumberField.textProperty().addListener(new ChangeListener<String>() {
            @Override
            public void changed(ObservableValue<? extends String> observable, String oldValue, 
                String newValue) {
                if (!newValue.matches("\\d*") || newValue.length() > 4) {
                    iterationsNumberField.setText(oldValue);
                }
            }
        });
        
        mutationRateField.textProperty().addListener(new ChangeListener<String>() {
            @Override
            public void changed(ObservableValue<? extends String> observable, String oldValue, 
                String newValue) {
                int newIntValue = 101;
                
                try {
                    newIntValue = Integer.parseInt(newValue);
                } catch (NumberFormatException e) {
                    if (newValue.isEmpty())
                        newIntValue = 1;
                } finally {
                    if (!newValue.matches("\\d*") || newIntValue > 100) {
                        mutationRateField.setText(oldValue);
                    }
                }
            }
        });
        
        observablePlaceList = FXCollections.observableArrayList();
        observableSuggestionsList = FXCollections.observableArrayList();
        
        placeListView.setCellFactory(lv -> {

            ListCell<String> cell = new ListCell<>();

            ContextMenu contextMenu = new ContextMenu();
            
            MenuItem startItem = new MenuItem();
            startItem.textProperty().bind(Bindings.format("Make \"%s\" the starting city", cell.itemProperty()));
            startItem.setOnAction(event -> {
                int i = placeListView.getItems().indexOf(cell.getItem());
                setStartingCity(i);
            });
            
            // Right click menu, button "info"
            MenuItem infoItem = new MenuItem();
            infoItem.textProperty().bind(Bindings.format("Info \"%s\"", cell.itemProperty()));
            infoItem.setOnAction(event -> {
                int i = placeListView.getItems().indexOf(cell.getItem());
                JFXDialogLayout content= new JFXDialogLayout();
                content.setHeading(new Text(placeList.get(i).address));
                
                GridPane dialogPane = new GridPane();
                dialogPane.setHgap(10);
                dialogPane.setVgap(10);
                
                Text infoLat = new Text("Latitude: " + placeList.get(i).latLng.lat);
                Text infoLng = new Text("Longitude: " + placeList.get(i).latLng.lng);
                Text infoGeneric = new Text("Look at it on Google Maps:");
                infoGeneric.setUnderline(true);
                
                Hyperlink link = new Hyperlink(placeList.get(i).link);
                link.setOnAction(new EventHandler<ActionEvent>() {
                    @Override
                    public void handle(ActionEvent t) {
                            hostServices.showDocument(link.getText());
                        }
                });
                dialogPane.add(infoLat, 0, 0);
                dialogPane.add(infoLng, 0, 1);
                dialogPane.add(infoGeneric, 0, 2);
                dialogPane.add(link, 1, 2);
                
                content.setBody(dialogPane);
                StackPane stackpane = new StackPane();
                JFXDialog dialog = new JFXDialog(stackPane, content, JFXDialog.DialogTransition.CENTER);
                JFXButton button = new JFXButton(" OK ");
                button.setOnAction(new EventHandler<ActionEvent>(){
                    @Override
                    public void handle(ActionEvent event){
                        dialog.close();
                    }
                });
                content.setActions(button);
                
                dialog.show();
            });
            
            // Right click menu, button "delete"
            MenuItem deleteItem = new MenuItem();
            deleteItem.textProperty().bind(Bindings.format("Delete \"%s\"", cell.itemProperty()));
            deleteItem.setOnAction(event -> {
                int i = placeListView.getItems().indexOf(cell.getItem());
                placeListView.getItems().remove(i);
                String removed = placeList.remove(i).address;
                
                if (observablePlaceList.isEmpty()) {
                    showOnMapButton.setDisable(true);
                }
                
                if (removed.equals(startingCity)) {
                    if (placeList.isEmpty())
                        setStartingCity(-1);
                    else
                        setStartingCity(0);
                }
                
                cityNumberText.setText(String.valueOf(Integer.parseInt(cityNumberText.getText())-1));
            });
            contextMenu.getItems().addAll(infoItem, deleteItem, startItem);

            cell.textProperty().bind(cell.itemProperty());

            cell.emptyProperty().addListener((obs, wasEmpty, isNowEmpty) -> {
                if (isNowEmpty) {
                    cell.setContextMenu(null);
                } else {
                    cell.setContextMenu(contextMenu);
                }
            });
            return cell ;
        });
    }
    
    public void setStartingCity(int index) {
        if (index < 0) {
            startingCity = "-";
            startingCityText.setText("-");
        }
        else {
            startingCity = placeList.get(index).address;
            
            if (startingCity.length() > 18) {
                startingCityText.setText(startingCity.substring(0, 15) + "...");
            }
            else {
                startingCityText.setText(placeList.get(index).address);
            }
            
            ArrayList<Place> tmp = new ArrayList<>();
            tmp.addAll(placeList.subList(index, placeList.size()));
            tmp.addAll(placeList.subList(0, index));
            
            placeList.clear();
            observablePlaceList.clear();
            
            for (Place p: tmp) {
                placeList.add(p);
                observablePlaceList.add(p.address);
            }
        }
    }
    
    public HostServices getHostServices() {
        return hostServices ;
    }

    public void setHostServices(HostServices hostServices) {
        this.hostServices = hostServices ;
    }
    
    private void clearSuggestions() {
        placeSuggestions.clear();
        observableSuggestionsList.clear();
        suggestionsListView.setItems(observableSuggestionsList);
    }
    
    private void setSuggestions(String address) {
        Task task = createTaskSearch(address);

        ProgressDialog dialog = new ProgressDialog(task);
        //dialog.setContentText("Searching for cities...");
        dialog.setTitle("Searching...");
        dialog.setHeaderText("Searching for cities...");

        new Thread(task).start();
        dialog.showAndWait();
    }
    
    private final ArrayList<String> suggestionsToAdd = new ArrayList<>();
    
    private Task createTaskSearch(String address) {
        return new Task() {
            @Override
            protected Void call() throws Exception {
                Place place;
                JOpenCageForwardRequest request = new JOpenCageForwardRequest(address);
                //request.setRestrictToCountryCode("it");
                request.setNoDedupe(true);
                request.setLimit(6);
                request.setNoAnnotations(true);
                //request.setNoRecord(true);

                JOpenCageResponse response = geocoder.forward(request);

                if (response.getResults().isEmpty())
                    System.err.println("Errore: Nessuna località trovata!");

                for (JOpenCageResult r: response.getResults()) {
                    place = new Place(r.getFormatted(), r.getGeometry().getLat(), r.getGeometry().getLng(), getLink(r));

                    if (!observableSuggestionsList.contains(place.toString())) {
                        suggestionsToAdd.add(place.toString());
                        placeSuggestions.add(place);
                    }
                }

                Platform.runLater(new Runnable() {
                    @Override
                    public void run() {
                        for (String place: suggestionsToAdd) {
                            observableSuggestionsList.add(place);
                        }
                        suggestionsListView.setItems(observableSuggestionsList);
                        suggestionsToAdd.clear();
                    }
                });
                
                return null;
            }
        };
    }
    
    private String getLink(JOpenCageResult r) {
        String result = r.getFormatted();
        
        result = result.replace(" ", "+");
        result = result.replace(",", "%2C");
        
        return "https://www.google.com/maps/search/?api=1&query=" + result;
    }
    
    @FXML
    public void showOnMapPressed() {
        hostServices.showDocument(getLinkFromList());
    }
    
    private String getLinkFromList() {
        String parameters = "";
        
        parameters += "&origin=" + placeList.get(0).URLaddress;
        
        parameters += "&waypoints=";
        for (int i = 1; i < placeList.size(); i++) {
            System.err.println(placeList.get(i).URLaddress);
            
            parameters += placeList.get(i).URLaddress;
            
            if (i < placeList.size()-1) {
                parameters +=  "%7C";
            }
        }
        
        parameters += "&destination=" + placeList.get(0).URLaddress;
        
        return "https://www.google.com/maps/dir/?api=1" + parameters;
    }
}
